# Shinobi Node.js Wrapper

## Official Node.js Wrapper for Shinobi API

> Documentation pending. Used internally at present. Available functions may change.

## Install

```
npm install node-shinobi
```


## `ShinobiAPI` Class

### Constructor: `constructor(shinobiEndpoint, apiKey, groupKey)`
- Initializes the `ShinobiAPI` instance with the Shinobi server endpoint, API key, and group key.
- **Parameters**:
  - `shinobiEndpoint` (string): The URL of the Shinobi server.
  - `apiKey` (string): The API key for authentication.
  - `groupKey` (string): The group key of the monitor group.

### Method: `async isServerAvailable()`
- Checks if the Shinobi server is available by making a `HEAD` request.
- **Returns**: `true` if the server is available, `false` otherwise.

### Method: `async checkAliveStatus(doPoll, onYes = () => {}, onNo = () => {})`
- Starts or stops a polling mechanism to check if the Shinobi server is available.
- **Parameters**:
  - `doPoll` (boolean): Whether to start or stop polling.
  - `onYes` (function): Callback function if the server is available.
  - `onNo` (function): Callback function if the server is unavailable.

### Method: `async configureMonitor(monitorId, data, action)`
- Sends a request to configure a monitor with specific data.
- **Parameters**:
  - `monitorId` (string): The ID of the monitor.
  - `data` (object): Configuration data for the monitor.
  - `action` (string, optional): Additional action parameter.
- **Returns**: The result of the configuration request.

### Method: `async getMonitor(monitorId)`
- Retrieves information about a specific monitor.
- **Parameters**:
  - `monitorId` (string, optional): The ID of the monitor.
- **Returns**: The monitor's details as a JSON object.

### Method: `async postVideo(filePath, ke, id, startTime, endTime)`
- Uploads a video file to the Shinobi server.
- **Parameters**:
  - `filePath` (string): Path to the video file.
  - `ke` (string): Group key.
  - `id` (string): Monitor ID.
  - `startTime` (Date): The start time of the video.
  - `endTime` (Date, optional): The end time of the video.
- **Returns**: The result of the video upload request.

### Method: `async monitorExist(monitorId)`
- Checks if a specific monitor exists in the Shinobi server.
- **Parameters**:
  - `monitorId` (string): The ID of the monitor.
- **Returns**: `true` if the monitor exists, `false` otherwise.


## Utility Functions

### `cleanStringForMonitorId(theString)`
- Cleans a string by removing special characters to generate a valid monitor ID.
- **Parameters**:
  - `theString` (string): The string to clean.
- **Returns**: The cleaned string.

### `getCameraTemplate(monitorConfigPartial)`
- Merges a partial monitor configuration with a predefined camera template.
- **Parameters**:
  - `monitorConfigPartial` (object): Partial configuration for the monitor.
- **Returns**: The merged monitor configuration object.

### `formatDateTime(dateTime)`
- Formats a `Date` object to a string suitable for use in filenames (replaces `:` and `.` with `-`).
- **Parameters**:
  - `dateTime` (Date): A `Date` object or string to format.
- **Returns**: A formatted string.


## Usage Examples

### Instantiating the `ShinobiAPI` Class
To create an instance of `ShinobiAPI`, you need to provide the Shinobi server endpoint, API key, and group key:

```
const {
    ShinobiAPI,
    formatDateTime,
    getCameraTemplate,
    cleanStringForMonitorId,
} = require('node-shinobi');

const shinobi = new ShinobiAPI('http://your-shinobi-endpoint', 'your-api-key', 'your-group-key');
```

### Checking if the Server is Available

You can use the `isServerAvailable()` method to check whether the Shinobi server is up and running:

```
const serverAvailable = await shinobi.isServerAvailable();
if (serverAvailable) {
    console.log('Shinobi server is available');
} else {
    console.log('Shinobi server is down');
}
```

### Polling for Server Availability

To continuously check if the server is available, you can use the `checkAliveStatus()` method with polling:

```
shinobi.checkAliveStatus(true,
    (status) => console.log('Server is alive', status),
    (status) => console.log('Server is down', status)
);
```

### Configuring a Monitor

To configure a monitor, use the `configureMonitor()` method:

```
const monitorConfig = {
    mid: 'monitor-id',
    mode: 'start',
    details: {
        fps: '30',
        resolution: '1920x1080'
    }
};

const response = await shinobi.configureMonitor('monitor-id', monitorConfig, 'edit');
console.log('Monitor configured:', response);
```

### Retrieving a Monitor's Information

To get information about a specific monitor, use the `getMonitor()` method:

```
const monitorInfo = await shinobi.getMonitor('monitor-id');
console.log('Monitor Info:', monitorInfo);
```

### Uploading a Video to Shinobi

You can use the `postVideo()` method to upload a video file to the Shinobi server:

```
const videoPath = '/path/to/video.mp4';
const startTime = new Date('2024-09-30T12:00:00');
const endTime = new Date('2024-09-30T12:30:00');

const uploadResponse = await shinobi.postVideo(videoPath, 'group-key', 'monitor-id', startTime, endTime);
console.log('Video uploaded:', uploadResponse);
```

### Checking if a Monitor Exists

To check whether a monitor exists in the Shinobi server:

```
const exists = await shinobi.monitorExist('monitor-id');
if (exists) {
    console.log('Monitor exists');
} else {
    console.log('Monitor does not exist');
}
```

### Utility Functions Usage

- Cleaning a String for Monitor ID:
```
const cleanId = cleanStringForMonitorId('my!@#monitor-id');
console.log('Cleaned Monitor ID:', cleanId);
```

- Getting a Camera Template:
```
const partialConfig = { mid: 'monitor-id', mode: 'start' };
const fullConfig = getCameraTemplate(partialConfig);
console.log('Full Camera Configuration:', fullConfig);
```

- Formatting a Date/Time:
```
const formattedDate = formatDateTime(new Date());
console.log('Formatted Date:', formattedDate);
```
