const path = require('path');
const fetch = require('node-fetch');
const cameraTemplate = require('./libs/cameraTemplateConfig.js');
const { createReadStream } = require('fs');
const {
    mergeDeep,
    fetchGet,
    fetchPost,
} = require('./libs/commonUtils.js');

class ShinobiAPI {
    constructor(shinobiEndpoint, apiKey, groupKey){
        this.status = {};
        this.pollTimer = null;
        this.lastPollCheck = false;
        this.apiPrefix = shinobiEndpoint.includes('://') ? shinobiEndpoint : `http://${shinobiEndpoint}`;
        this.apiPrefix = `${this.apiPrefix.endsWith('/') ? this.apiPrefix : this.apiPrefix + '/'}${apiKey}/`;
        this.groupKey = groupKey;
    }
    async isServerAvailable() {
        try {
            const url = `${this.apiPrefix}videos/${this.groupKey}`;
            const response = await fetch(url, { method: 'HEAD' });
            this.status.ok = response.ok;
            this.status.time = new Date();
            return response.ok;
        } catch (error) {
            console.error(`${this.apiPrefix} is not available.`);
            return false;
        }
    }
    async checkAliveStatus(doPoll, onYes = () => {}, onNo = () => {}) {
        clearInterval(this.pollTimer);
        if (doPoll) {
            this.pollTimer = setInterval(async () => {
                const isOk = await this.isServerAvailable();
                if (this.lastPollCheck !== isOk) {
                    this.lastPollCheck = isOk;
                    (isOk ? onYes : onNo)(this.status);
                }
            }, 1000 * 60 * 5);
        }
    }
    async configureMonitor(monitorId, data, action) {
        const theConfigToPost = mergeDeep({}, await this.getMonitor(monitorId) || cameraTemplate, data);
        const url = `${this.apiPrefix}configureMonitor/${this.groupKey}/${monitorId}${action ? `/${action}` : ''}`;
        try {
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    data: JSON.stringify(theConfigToPost),
                }),
            });
            const result = await response.text();
            return result;
        } catch (error) {
            console.error(error);
            throw new Error(`Error: ${error.message}`);
        }
    }
    async getMonitor(monitorId) {
        const url = `${this.apiPrefix}monitor/${this.groupKey}${monitorId ? `/${monitorId}` : ''}`;
        try {
            const response = await fetch(url);
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            const result = await response.json();
            return result;
        } catch (error) {
            console.error(new Error(error.message))
            return null
        }
    }
    async postVideo(filePath, ke, id, startTime, endTime) {
        const url = `${this.apiPrefix}videos/${ke}/${id}`;
        const filename = `${formatDateTime(startTime)}.mp4`;
        const formData = new FormData();
        formData.append('video', createReadStream(filePath), filename);
        if (endTime) {
            formData.append('endTime', formatDateTime(endTime));
        }
        try {
            const response = await fetch(url, {
                method: 'POST',
                body: formData
            });
            if (!response.ok) {
                throw new Error(`Failed to upload video. Status: ${response.status}`);
            }
            const result = await response.json();
            return result;
        } catch (error) {
            this.status.ok = true;
            console.error(`Error uploading video: ${error.message}`);
            throw error;
        }
    }
    async monitorExist(monitorId) {
        const rows = await this.getMonitor(monitorId);
        return !!rows[0];
    }
}

function cleanStringForMonitorId(theString) {
    return theString.replace(/[`~!@#$%^&*()_|+\-=?;:'",.<>\{\}\[\]\\\/]/gi, '');
}

function getCameraTemplate(monitorConfigPartial) {
    const monitorConfig = mergeDeep({}, cameraTemplate, monitorConfigPartial);
    return monitorConfig;
}

function formatDateTime(dateTime) {
    return dateTime instanceof Date
        ? dateTime.toISOString().replace(/[:.]/g, '-')
        : dateTime;
}

module.exports = {
    ShinobiAPI,
    formatDateTime,
    getCameraTemplate,
    cleanStringForMonitorId,
};
